<?php

use Composer\Config;
use Composer\IO\NullIO;
use Composer\Plugin\PluginEvents;
use Composer\Plugin\PreFileDownloadEvent;
use Composer\Util\HttpDownloader;
use RalphJSmit\Packages\Plugin;

beforeEach(function () {
    $this->phpOsFamily = PHP_OS_FAMILY;
});

it('can generate an identifier and construct the URL', function () {
    $dir = '/Users/ralphjsmit/Code/Sites/test-project/vendor/ralphjsmit/packages/src';

    $plugin = new Plugin(directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => null);

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1($dir) . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: .
    Executed shell command [cat /proc/self/cgroup 2>&1] with output: .
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=0' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});

it('can generate an identifier and construct the URL for Windows', function () {
    $dir = '\Users\ralphjsmit\Code\Sites\test-project\vendor\ralphjsmit\packages\src';

    $plugin = new Plugin(directorySeparator: '\\', directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => null);

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1($dir) . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: .
    Executed shell command [cat /proc/self/cgroup 2>&1] with output: .
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=0' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});

it('can generate an identifier and construct the URL for Laravel Envoyer', function () {
    $dir = '/Users/ralphjsmit/Code/Sites/test-project/releases/20240101140000/vendor/ralphjsmit/packages/src';

    $plugin = new Plugin(directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => null);

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1('/Users/ralphjsmit/Code/Sites/test-project/releases/{release}/vendor/ralphjsmit/packages/src') . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    Directory contains "releases" folder. Assuming Envoyer structure.
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: .
    Executed shell command [cat /proc/self/cgroup 2>&1] with output: .
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=0' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});

it('can generate an identifier and construct the URL for Laravel Envoyer on Windows based server', function () {
    $dir = '\Users\ralphjsmit\Code\Sites\test-project\releases\20240101140000\vendor\ralphjsmit\packages\src';

    $plugin = new Plugin(directorySeparator: '\\', directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => null);

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1('\Users\ralphjsmit\Code\Sites\test-project\releases\{release}\vendor\ralphjsmit\packages\src') . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    Directory contains "releases" folder. Assuming Envoyer structure.
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: .
    Executed shell command [cat /proc/self/cgroup 2>&1] with output: .
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=0' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});

it('can determine whether Docker is running based on control groups if docker env is not present', function () {
    $dir = '/Users/ralphjsmit/Code/Sites/test-project/vendor/ralphjsmit/packages/src';

    $plugin = new Plugin(directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => match ($command) {
        'ls -la ../../../../. 2>&1' => '',
        'cat /proc/self/cgroup 2>&1' => 'docker',
    });

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1('/Users/ralphjsmit/Code/Sites/test-project/vendor/ralphjsmit/packages/src') . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: .
    Executed shell command [cat /proc/self/cgroup 2>&1] with output: docker.
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=1' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});

it('can determine whether Docker is running based on docker env', function () {
    $dir = '/Users/ralphjsmit/Code/Sites/test-project/vendor/ralphjsmit/packages/src';

    $dockerEnvOutput = '-rw-r--r--   1 ralphjsmit  staff    676 Jan 10 10:01 .dockerenv';

    $plugin = new Plugin(directoryResolver: fn () => $dir, shellExecutor: fn (string $command) => match ($command) {
        'ls -la ../../../../. 2>&1' => $dockerEnvOutput,
    });

    $preFileDownloadEvent = new PreFileDownloadEvent(
        name: PluginEvents::PRE_FILE_DOWNLOAD,
        httpDownloader: new HttpDownloader(new NullIO(), new Config()),
        processedUrl: 'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip',
        type: 'package'
    );

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe('https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip');

    $plugin->onPreFileDownload($preFileDownloadEvent);

    $expectedIdentifier = urlencode(gethostname() . '|' . sha1('/Users/ralphjsmit/Code/Sites/test-project/vendor/ralphjsmit/packages/src') . '|' . ('test-project'));

    $expectedLog = urlencode(<<<OUTPUT
    PHP_OS_FAMILY: {$this->phpOsFamily}
    Executed shell command [ls -la ../../../../. 2>&1] with output: {$dockerEnvOutput}.
    OUTPUT);

    expect($preFileDownloadEvent)
        ->getProcessedUrl()->toBe(
            'https://satis.ralphjsmit.com/dist/ralphjsmit/laravel-filament-media-library/ralphjsmit-laravel-filament-media-library-5aa15ac21255b3b617c3d14d116200e469b8e7af-zip-6bd925.zip?id=' . $expectedIdentifier . '&docker=1' . '&ralphjsmit-packages-version=' . Plugin::PLUGIN_VERSION . '&log=' . $expectedLog
        );
});
