<?php

namespace RalphJSmit\Packages;

use Closure;
use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginEvents;
use Composer\Plugin\PluginInterface;
use Composer\Plugin\PreFileDownloadEvent;

class Plugin implements EventSubscriberInterface, PluginInterface
{
    public const PLUGIN_VERSION = '1.2.0';

    protected Closure $directoryResolver;

    protected Closure $shellExecutor;

    public function __construct(
        protected string $directorySeparator = DIRECTORY_SEPARATOR,
        ?Closure $directoryResolver = null,
        ?Closure $shellExecutor = null,
    ) {
        $this->directoryResolver = $directoryResolver ?? function (): string {
            return __DIR__;
        };

        $this->shellExecutor = $shellExecutor ?? function (string $command): ?string {
            // Redirect any errors in stderr to stdout (https://stackoverflow.com/a/6780318)...
            if (! str_ends_with($command, ' 2>&1')) {
                $command .= ' 2>&1';
            }

            return @shell_exec($command);
        };
    }

    public static function getSubscribedEvents(): array
    {
        return [
            PluginEvents::PRE_FILE_DOWNLOAD => [
                [
                    'onPreFileDownload',
                    9999,
                ],
            ],
        ];
    }

    public function activate(Composer $composer, IOInterface $io): void
    {
        //
    }

    public function deactivate(Composer $composer, IOInterface $io): void
    {
        //
    }

    public function uninstall(Composer $composer, IOInterface $io): void
    {
        //
    }

    /**
     * Note for future self: update the `PLUGIN_VERSION` constant before releasing.
     */
    public function onPreFileDownload(PreFileDownloadEvent $event): void
    {
        $processedUrl = $event->getProcessedUrl();

        if (! str_contains($processedUrl, 'ralphjsmit')) {
            return;
        }

        $directory = ($this->directoryResolver)();

        if (str_contains($directory, 'releases')) {
            $directorySanitized = $this->getEnvoyerDirectorySanitized($directory);
            $directoryName = $this->getEnvoyerDirectoryName($directory);
        } else {
            $directorySanitized = $directory;
            $directoryName = $this->getDefaultDirectoryName($directory);
        }

        $identifier = gethostname() . '|' . sha1($directorySanitized) . '|' . $directoryName;

        // Modifying this code is against the product license. Just buy the dang thing and save yourself the effort.
        $event->setProcessedUrl(
            $processedUrl . '?id=' . urlencode($identifier) . '&docker=' . (int) $this->isDocker() . '&ralphjsmit-packages-version=' . static::PLUGIN_VERSION
        );
    }

    protected function getEnvoyerDirectorySanitized(string $directory): string
    {
        $directorySeparator = $this->directorySeparator;

        if ($directorySeparator === '\\') {
            $directorySeparator = '\\\\';
        }

        return preg_replace(
            '#' . $directorySeparator . 'releases' . $directorySeparator . '.*?' . $directorySeparator . 'vendor' . $directorySeparator . '#',
            "{$this->directorySeparator}releases{$this->directorySeparator}{release}{$this->directorySeparator}vendor{$this->directorySeparator}",
            $directory
        );
    }

    protected function getEnvoyerDirectoryName(string $directory): string
    {
        // Str::before() implementation...
        $directoryBeforeReleases = strstr($directory, 'releases', true);

        if ($directoryBeforeReleases === false) {
            $directoryBeforeReleases = $directory;
        }

        // Trim the trailing directory separator.
        $directoryBeforeReleases = rtrim($directoryBeforeReleases, $this->directorySeparator);

        // Str::afterLast() implementation...
        $positionLastDirectorySeparator = strrpos($directoryBeforeReleases, (string) $this->directorySeparator);

        if ($positionLastDirectorySeparator === false) {
            return $directoryBeforeReleases;
        }

        return substr($directoryBeforeReleases, $positionLastDirectorySeparator + strlen($this->directorySeparator));
    }

    protected function getDefaultDirectoryName(string $directory): string
    {
        $directorySeparator = $this->directorySeparator;

        // Windows uses backslashes as directory separators. If we use a backslash in the string for regex, we need to escape this one double.
        if ($directorySeparator === '\\') {
            $directorySeparator = '\\\\';
        }

        preg_match(
            '#' . $directorySeparator . '([^' . $directorySeparator . ']+)' . $directorySeparator . 'vendor' . $directorySeparator . '#',
            $directory,
            $matches
        );

        return $matches[1] ?? $directory;
    }

    protected function isDocker(): bool
    {
        // Redirect stderr to stdout with 2>&1 (https://stackoverflow.com/a/6780318)...
        $controlGroups = ($this->shellExecutor)('cat /proc/self/cgroup');

        if (! $controlGroups) {
            return false;
        }

        // Example output on Hetzner:
        // forge@ralphjsmit-composer-test:~$ cat /proc/self/cgroup
        // 0::/user.slice/user-1000.slice/session-46132.scope.
        //
        // Example output on Docker:
        // 12:rdma:/
        // 11:perf_event:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 10:freezer:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 9:memory:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 8:cpuset:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 7:devices:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 6:net_cls,net_prio:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 5:hugetlb:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 4:pids:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 3:cpu,cpuacct:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 2:blkio:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 1:name=systemd:/docker/a2ffe0e97ac22657a2a023ad628e9df837c38a03b1ebc904d3f6d644eb1a1a81
        // 0::/system.slice/containerd.service
        // Source: https://stackoverflow.com/questions/69002675/on-debian-11-bullseye-proc-self-cgroup-inside-a-docker-container-does-not-sho
        $controlGroups = array_filter(
            explode(PHP_EOL, $controlGroups)
        );

        foreach ($controlGroups as $controlGroup) {
            if (str_contains($controlGroup, 'docker')) {
                return true;
            }
        }

        return false;
    }
}
